/*
** file: ngrep.c
**
** UC: 21111 - Sistemas Operativos
** e-folio B 2024-25
** Simular comando grep -n
** Versão template (VPL)
**
** Botão "Run"    <=> Utilização com dados do utilizador em terminal:
** ./ngrep
** Botão "Evaluate" <=> Utilização com redireção de ficheiros automática:
** ./ngrep <input01.txt >output01.txt
** Compilação VPL (auto):
** gcc -Wall -D_REENTRANT -std=c99 -o ngrep ngrep.c –lpthread
**
** Aluno: 2203524 - Carlos Miguel Faria Martins da Costa
*/

#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include <sys/types.h>
#include <unistd.h>

#define MAX_LINE_LENGTH 1024 // Define o tamanho máximo de uma linha lida do ficheiro

typedef struct FoundLineInfo {
    int line_number; // Número da linha onde o padrão foi encontrado
    char *line_text; // Conteúdo da linha onde o padrão foi encontrado
} FoundLineInfo;

typedef struct TaskData {
    int task_id;      // Identificador único da tarefa (thread)
    long start_byte;  // Byte de início da porção do ficheiro a ser processada por esta thread
    long end_byte;    // Byte de fim da porção do ficheiro a ser processada por esta thread
    char *string;     // A string de pesquisa
    char *filename;   // O nome do ficheiro a ser pesquisado
    int lines_found;  // Número de linhas encontradas por esta thread
    FoundLineInfo *found_lines_info; // Array dinâmico para armazenar informações sobre as linhas encontradas
} TaskData;

int global_line_count = 0;      // Contador global de linhas que contêm o padrão
pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER; // Mutex para proteger o acesso à variável global global_line_count e à estrutura de dados de resultados

/**
 * @brief Encontra o início da primeira linha dentro de um intervalo de bytes e calcula o número da linha inicial.
 *
 * @param file Ponteiro para o ficheiro.
 * @param start_byte O byte de início do intervalo.
 * @param initial_line_number Ponteiro para um inteiro onde o número da linha inicial será armazenado.
 * @return 0 em caso de sucesso, 1 em caso de erro ao ler o ficheiro.
 */
int find_start_line(FILE *file, long start_byte, int *initial_line_number) {
    if (start_byte == 0) {
        *initial_line_number = 1; // Se começar do início do ficheiro, o número da primeira linha é 1
        return 0;
    }

    fseek(file, start_byte, SEEK_SET); // Posiciona o ponteiro do ficheiro no byte de início
    char c;
    long current_byte = start_byte;
    int lines_skipped = 0;
    int newline_found = 0;

    // Move back until the beginning of a line or the start of the file
    while (current_byte > 0) {
        fseek(file, --current_byte, SEEK_SET); // Retrocede um byte
        if (fread(&c, 1, 1, file) != 1) {
            return 1; // Erro ao ler o ficheiro
        }
        if (c == '\n') {
            newline_found = 1; // Encontrou uma nova linha
            break;
        }
    }

    // Count the lines before the start byte
    fseek(file, 0, SEEK_SET); // Volta ao início do ficheiro
    char line[MAX_LINE_LENGTH];
    while (ftell(file) < start_byte) {
        fgets(line, sizeof(line), file); // Lê uma linha
        lines_skipped++; // Incrementa o contador de linhas
    }

    *initial_line_number = lines_skipped + newline_found; // Calcula o número da linha inicial
    return 0;
}

/**
 * @brief Encontra o byte de fim para cada thread, garantindo que termina no final de uma linha.
 *
 * @param file Ponteiro para o ficheiro.
 * @param end_byte O byte de fim desejado.
 * @return O byte de fim ajustado para o final de uma linha.
 */
long find_end_byte(FILE *file, long end_byte) {
    if (end_byte == -1) { // End of file
        fseek(file, 0, SEEK_END);
        return ftell(file); // Retorna o tamanho total do ficheiro
    }

    fseek(file, end_byte, SEEK_SET); // Posiciona o ponteiro do ficheiro no byte de fim
    char c;
    while (fread(&c, 1, 1, file) == 1) {
        if (c == '\n') {
            return ftell(file); // Retorna a posição após a nova linha
        }
    }
    return ftell(file); // Reached EOF
}

/**
 * @brief Função executada por cada thread para pesquisar o padrão no ficheiro.
 *
 * @param arg Ponteiro para a estrutura TaskData que contém os dados da tarefa para a thread.
 * @return NULL (o valor de retorno é ignorado para threads).
 */
void *worker_task(void *arg) {
    TaskData *data = (TaskData *)arg; // Converte o argumento para o tipo TaskData
    FILE *file = fopen(data->filename, "r"); // Abre o ficheiro
    if (!file) {
        perror("Erro ao abrir o ficheiro");
        pthread_exit(NULL); // Termina a thread em caso de erro
    }

    int current_line_number = 1;
    if (find_start_line(file, data->start_byte, &current_line_number) != 0) {
        fclose(file);
        pthread_exit(NULL);
    }

    fseek(file, data->start_byte, SEEK_SET); // Posiciona o ponteiro do ficheiro no byte de início

    char *line = NULL;
    size_t len = 0;
    ssize_t read;
    long current_byte = ftell(file);

    while ((read = getline(&line, &len, file)) != -1) { // Lê linha a linha
        if (data->end_byte != -1 && current_byte >= data->end_byte) {
            break; // Se o byte atual exceder o byte de fim atribuído a esta thread, parar de ler
        }
        if (strstr(line, data->string)) { // Procura a string no conteúdo da linha
            pthread_mutex_lock(&mutex);
            global_line_count++; // Incrementa o contador global de linhas encontradas (acesso protegido pelo mutex)
            pthread_mutex_unlock(&mutex);

            data->found_lines_info = realloc(data->found_lines_info, (data->lines_found + 1) * sizeof(FoundLineInfo)); // Realoca memória para armazenar informações da linha encontrada
            if (!data->found_lines_info) {
                perror("Erro ao alocar memória");
                free(line);
                fclose(file);
                pthread_exit(NULL);
            }
            data->found_lines_info[data->lines_found].line_number = current_line_number; // Armazena o número da linha
            data->found_lines_info[data->lines_found].line_text = strdup(line); // Armazena o conteúdo da linha
            data->lines_found++; // Incrementa o contador de linhas encontradas por esta thread
        }
        current_line_number++; // Incrementa o número da linha atual
        current_byte = ftell(file); // Obter a posição atual no ficheiro
    }

    free(line);
    fclose(file);
    pthread_exit(data); // Termina a thread
}

/**
 * @brief Função principal do programa.
 *
 * @param argc Número de argumentos da linha de comando.
 * @param argv Array de strings contendo os argumentos da linha de comando.
 * @return 0 em caso de sucesso, 1 em caso de erro.
 */
int main(int argc, char *argv[]) {
    char input[MAX_LINE_LENGTH];
    if (fgets(input, sizeof(input), stdin) == NULL) { // Lê a entrada do utilizador
        perror("Erro ao ler a entrada");
        return 1;
    }
    input[strcspn(input, "\n")] = 0; // Remove a nova linha do final da entrada

    char *string_start = strchr(input, '"'); // Encontra o início da string de pesquisa
    if (string_start == NULL) {
        fprintf(stderr, "Erro: String de pesquisa não encontrada.\n");
        return 1;
    }
    string_start++; // Avança para o primeiro caractere da string
    char *string_end = strchr(string_start, '"'); // Encontra o fim da string de pesquisa
    if (string_end == NULL) {
        fprintf(stderr, "Erro: Fim da string de pesquisa não encontrado.\n");
        return 1;
    }
    *string_end = '\0'; // Termina a string de pesquisa
    char *string = strdup(string_start); // Duplica a string de pesquisa

    char *filename = strtok(string_end + 1, " "); // Obtém o nome do ficheiro
    if (filename == NULL) {
        fprintf(stderr, "Erro: Nome do ficheiro não encontrado.\n");
        free(string);
        return 1;
    }

    char *nt_str = strtok(NULL, " "); // Obtém o número de tarefas (threads)
    if (nt_str == NULL) {
        fprintf(stderr, "Erro: Número de tarefas não encontrado.\n");
        free(string);
        return 1;
    }
    int nt = atoi(nt_str); // Converte o número de tarefas para inteiro

    if (!string || !filename || nt < 1 || nt > 9) { // Valida os parâmetros de entrada
        printf("Parâmetros inválidos\n");
        free(string);
        return 1;
    }

    FILE *file = fopen(filename, "r"); // Abre o ficheiro
    if (!file) {
        perror("Erro ao abrir o ficheiro");
        free(string);
        return 1;
    }

    fseek(file, 0, SEEK_END); // Vai para o fim do ficheiro
    long file_size = ftell(file); // Obtém o tamanho do ficheiro
    fclose(file);

    if (file_size < nt) {
        nt = 1; // Se o ficheiro for menor que o número de threads, usa apenas 1 thread
    }

    printf("Procurar linhas com o texto \"%s\" no ficheiro %s com %d tarefas\n", string, filename, nt);

    pthread_t threads[nt]; // Array de threads
    TaskData task_data[nt]; // Array de dados das tarefas

    long base_size = file_size / nt; // Tamanho base de cada porção do ficheiro
    long remainder = file_size % nt; // Restante do tamanho do ficheiro
    long current_start = 0; // Byte de início da porção atual

    for (int i = 0; i < nt; i++) {
        task_data[i].task_id = i; // Atribui o ID da tarefa
        task_data[i].start_byte = current_start; // Define o byte de início
        FILE *temp_file = fopen(filename, "r"); // Abre um ficheiro temporário para calcular o byte de fim
        if (!temp_file) {
            perror("Erro ao abrir o ficheiro temporário");
            free(string);
            return 1;
        }
        long chunk_size = base_size + (i < remainder ? 1 : 0); // Calcula o tamanho da porção
        task_data[i].end_byte = current_start + chunk_size; // Calcula o byte de fim
        task_data[i].end_byte = find_end_byte(temp_file, task_data[i].end_byte); // Ajusta o byte de fim para o final de uma linha
        fclose(temp_file);
        if (i == nt - 1) {
            task_data[i].end_byte = -1; // A última thread lê até o fim do ficheiro
        }
        task_data[i].string = string; // Atribui a string de pesquisa
        task_data[i].filename = filename; // Atribui o nome do ficheiro
        task_data[i].lines_found = 0; // Inicializa o contador de linhas encontradas
        task_data[i].found_lines_info = NULL; // Inicializa o array de informações das linhas encontradas

        current_start = task_data[i].end_byte; // Atualiza o byte de início para a próxima thread

        if (pthread_create(&threads[i], NULL, worker_task, &task_data[i]) != 0) { // Cria a thread
            perror("Erro ao criar thread");
            free(string);
            return 1;
        }
    }

    for (int i = 0; i < nt; i++) {
        if (pthread_join(threads[i], NULL) != 0) { // Espera que a thread termine
            perror("Erro ao juntar thread");
        }
    }

    printf("Encontradas %d linhas\n", global_line_count); // Imprime o número total de linhas encontradas

    for (int i = 0; i < nt; i++) {
        for (int j = 0; j < task_data[i].lines_found; j++) {
            printf("(.) %d:%s", task_data[i].found_lines_info[j].line_number, task_data[i].found_lines_info[j].line_text); // Imprime o número da linha e o conteúdo
            free(task_data[i].found_lines_info[j].line_text); // Libera a memória alocada para o conteúdo da linha
        }
        free(task_data[i].found_lines_info); // Libera a memória alocada para o array de informações das linhas
    }

    free(string); // Libera a memória alocada para a string de pesquisa

    return 0;
}
